; Check if the script was called with at least one parameter (the file path)
global FileEnding

if (0 == A_Args.Length())
{
	FileEnding := "_manual." ; The Original File is not overwritten when the file has been selected manually, instead a file with extenstion _manual is saved
	FileSelectFile, selectedFile,, Please select a file,, *.*
	if (!FileExist(selectedFile))
	{
		MsgBox, % "File does not exist."
		return
	}
} else 
{
	selectedFile := A_Args[1]  ; Get the file path from the first command line parameter
	FileEnding := "." ; The file-name is not modified when used as integrated Postprozessor
}

FileRead, fileContent, %selectedFile%
if (ErrorLevel)
{
    MsgBox, % "Failed to read the file."
    return
}

currentMode := "" ; Initialize the mode variable
FirstTool := true
global JustChanged := false
PositionFlagM := "High" ; Assume starting position for Milling is High
PositionFlagP := "Low" ; Assume starting position for PenPlotting is Low
PositionFlagV := "Low" ; Assume starting position for VialPlotting is Low
global xValue := "0"
global yValue := "0"

newFileContent := "(BHK-postprocessing)`n(M03 Red LED on, M04 Red LED off)`n(M08 Green LED on, M09 Green LED off)`n(MBC-R = New Reference Point)`n(MBC-M = Milling)`n(MBC-P = Pen Plotting)`n(MBC-C = Cutting)`n(MBC-S = SyringePlotting)`n(MBC-L = Laser operation)"

Loop, parse, fileContent, `n, `r
{
    line := A_LoopField
    if (RegExMatch(line, "^M06") || (RegExMatch(line, "MBC-") && FirstTool))
    {
		FirstTool := false
        ; Replace M06(Tool Change) with M0 (Pause code)
        line := line . "debug"
		line := StrReplace(line, "M06", "(")
		line := line . ")`nG00 X" . xValue . " Y" . yValue . "`nM0"

        ; Check if the line contains "MBC-" (All Mobi-C tools in ESTLCAM shall be named with MBC- prefix in order to trigger the correct post-prozessing 
        if (InStr(line, "MBC-"))
        {
            mode := SubStr(line, InStr(line, "MBC-") + 4, 1)
            if (mode = "R") ;Used for setting a new Reference Point, the selected point will be "0,0"
            {
                currentMode := "Reference"
                newFileContent .= "( Mode: New Reference Point)`n"
            }
			else if (mode = "M")
            {
                currentMode := "Milling"
                newFileContent .= "( Mode: Milling)`nM04`nM09`n"
            }
            else if (mode = "S")
            {
                currentMode := "SyringePlotting"
                newFileContent .= "( Mode: SyringePlotting)`n"
            }
            else if (mode = "P")
            {
                currentMode := "PenPlotting"
                newFileContent .= "( Mode: PenPlotting)`n"
            }            
			else if (mode = "C")
            {
                currentMode := "Cutting"
                newFileContent .= "( Mode: DragKnife Cutting)`n"
            }
			else if (mode = "L")
            {
                currentMode := "Laser"
                newFileContent .= "( Mode: Laser)`n"
            }
            else
            {
                currentMode := "" ; Reset if mode is not recognized
                newFileContent .= "( Mode: not recognized)`n"
            }
        }
        else
        {
            currentMode := "" ; Reset if "MBC-" is not found
            newFileContent .= "( Mode: standard...no code change)`n"
        }
    }

    if (currentMode = "Milling")
    {
        newFileContent := ProcessMillingFunction(line, newFileContent)
    }
    else if (currentMode = "PenPlotting" || currentMode = "Cutting")
    {
        newFileContent := ProcessPenPlottingFunction(line, newFileContent)
    }
    else if (currentMode = "SyringePlotting")
    {
        newFileContent := ProcessSyringePlottingFunction(line, newFileContent)
    }
    else if (currentMode = "Laser")
    {
        newFileContent .= line . "`n"  ;Presently no code change for Laser but special commands in ESTLCAM
    }
	else if (currentMode = "Reference")
    {
        newFileContent := ProcessNewReferenceFunction(line, newFileContent)
    }
    else
    {
        newFileContent .= line . "`n"
    }
}


SaveFile(newFileContent, selectedFile)

ProcessNewReferenceFunction(line, newFileContent)
{
    global xValue, yValue 
	; Delete lines starting with G01, M03, M04, M05, M08, or M09
    if (RegExMatch(line, "^(M03|M04|M05|M08|M09|G01|G00 Z)"))
        return newFileContent

    ; Check for G00 lines with x & y coordinates (Drill positioning)
    if (SubStr(line, 1, 3) = "G00" && RegExMatch(line, "X([\d\.]+)", xMatch) && RegExMatch(line, "Y([\d\.]+)", yMatch) && RegExMatch(line, "Z([\d\.]+)"))
    {
        ; Extract X and Y values
        xValue := xMatch1
        yValue := yMatch1
        
        ; Construct the new line with G92 and the extracted X and Y values
        newLine := "G92 X" . xValue . " Y" . yValue
        newFileContent .= newLine . "`n"
    }
    else
    {
        ; Add the current line after any inserted lines
        newFileContent .= line . "`n"
    }
    
    return newFileContent
}


ProcessMillingFunction(line, newFileContent)
{
    global PositionFlagM, JustChanged
    ; Delete lines starting with M04, M08, or M09
    if (RegExMatch(line, "^(M03|M04|M08|M09)"))
        return newFileContent

    ; Check for lines containing "Z-" when JustChanged is false
    if (JustChanged = false && RegExMatch(line, "Z-\d+(\.\d+)?")) ; Code to switch laser on
    {
        newFileContent .= "M03`nM00`nM04`n" ; green LED on - pause - green LED off
        PositionFlagM := "High"
        JustChanged := true
    }
	else if (not RegExMatch(line, "Z-\d+(\.\d+)?"))
	{
		JustChanged := false
	}

    ; Check for lines containing "Z" followed by a positive number
    if (PositionFlagM = "High" && RegExMatch(line, "Z\d+(\.\d+)?")) ; Code to switch laser off
    {
        newFileContent .= "M08`nM00`nM09`n" ; red LED on - pause - red LED off
        PositionFlagM := "Low"
    }


    ; Add the current line after any inserted lines
    newFileContent .= line . "`n"
    return newFileContent
}



ProcessPenPlottingFunction(line, newFileContent)
{
    global PositionFlagP
 	; Remove any string starting with "S" followed by a number, anywhere in the line
    line := RegExReplace(line, "S\d+", "")

    ; Delete lines starting with M3, M04, M05, M08 or M09 since these commands will be used to controll the LEDs
    if (RegExMatch(line, "^(M03|M04|M05|M08|M09)"))
        return newFileContent

    ; Replace M03 with M04 in lines containing M03 since not spindle direction control is supported
    ;StringReplace, line, line, M03, M04, All

    ; Check for G00 lines with Z followed by a number >= 1
    if (SubStr(line, 1, 3) = "G00" && RegExMatch(line, "Z([1-9]\d*)") && PositionFlagP = "Low")
    {
        newFileContent .= "M03`nS14`n"
        PositionFlagP := "High"
    }

    ; Check for G01 lines containing "Z-" when PositionFlagP is High
    if (SubStr(line, 1, 2) = "G0" && InStr(line, "Z-") && PositionFlagP = "High")
    {
        newFileContent .= "S8`nM05`n"
        PositionFlagP := "Low"
    }

    ; Add the current line after any inserted lines
    newFileContent .= line . "`n"
    return newFileContent
}

ProcessSyringePlottingFunction(line, newFileContent)
{
    global PositionFlagV
    ; Remove any string starting with "S" followed by a number, anywhere in the line
    line := RegExReplace(line, "S\d+", "")

    ; Delete lines starting with M08 or M09
    if (RegExMatch(line, "^(M08|M09)"))
        return newFileContent

    ; Replace M03 with M04 in lines containing M03
    StringReplace, line, line, M03, M04, All

    ; Check for G00 lines with Z followed by a number >= 1
    if (SubStr(line, 1, 3) = "G00" && RegExMatch(line, "Z([1-9]\d*)") && PositionFlagV = "Low")
    {
        newFileContent .= "M04`nS30`n"
        PositionFlagV := "High"
    }

    ; Check for G01 lines containing "Z-" when PositionFlagV is High
    if (SubStr(line, 1, 2) = "G0" && InStr(line, "Z-") && PositionFlagV = "High")
    {
        newFileContent .= "S36`nM03`n"
        PositionFlagV := "Low"
    }

    ; Add the current line after any inserted lines
    newFileContent .= line . "`n"
    return newFileContent
}

SaveFile(newFileContent, selectedFile)
{
	global FileEnding
    ; Extract the directory, filename without extension, and extension
    SplitPath, selectedFile, name, dir, ext, name_no_ext

    ; Construct the new file name
    newFileName := dir . "\" . name_no_ext . FileEnding . ext

    ; Delete the file if it exists to ensure it's replaced
    IfExist, %newFileName%
    {
        FileDelete, %newFileName%
    }

    ; Save the final content to the new file
    FileAppend, %newFileContent%, %newFileName%

    MsgBox, % "File processing complete. New file saved as '" . newFileName . "'."
}
